/***************************************************************************//**
 *
 * @page mainl Mainloop Concept
 *
 * @section mconcept Mainloop
 * The AudioManager comes with a build in main input processing loop which can be utilized
 * by the plug-ins to serve their needs of communication and thread-safe calling.
 * It can monitor inputs like sockets, files, timers and signals, all aligned to a single thread.
 * The main loop is implemented in class CAmSocketHandler and works like this:\n
 *
 * @image html Mainloop.png
 *
 * @section sec Using the Mainloop
 *
 * @copydoc am::CAmSocketHandler
 *
 * @section secThreadAlign Utilizing The Mainloop as Threadsafe Call Method
 *
 * The AudioManager itself is single-threaded, so any calls from other threads inside the plugins
 * directly to the interfaces are forbidden, otherwise the behavior is undefined. The reason for
 * this is that command and routing plugins are often only communication interfaces that are
 * ideally used with the am::CAmSocketHandler.\n
 *
 * @copydoc am::V2::CAmSerializer
 *
 * @subsection async Asynchronous calls
 * @image html Deferred_Call_async.png
 * @subsection sync Synchronous calls
 * @image html Deferred_Call_sync.png
 *
 *//***********************************************************************//**
 *  @file CAmSocketHandler.h
 *
 *  @copybrief am::CAmSocketHandler
 *
 *  SPDX license identifier: MPL-2.0
 *
 *  @copyright (C) 2012, BMW AG
 *
 *  @author  Christian Linke,    <christian.linke@bmw.de>            BMW 2011,2012
 *  @author  Aleksandar Donchev, <aleksander.donchev@partner.bmw.de> BMW 2017
 *  @author  Martin Koch,        <mkoch@de.adit-jv.com>              ADIT 2020
 *
 *  @copyright
 *  This Source Code Form is subject to the terms of the
 *  Mozilla Public License, v. 2.0. If a  copy of the MPL was not distributed with
 *  this file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 *  For further information see http://www.genivi.org/.
 *
 *//***************************************************************************/


#ifndef SOCKETHANDLER_H_
#define SOCKETHANDLER_H_

#include <sys/socket.h>
#include <stdint.h>
#include <sys/poll.h>
#include <unistd.h>
#include <list>
#include <map>
#include <set>
#include <signal.h>
#include <vector>
#include <functional>
#include <pthread.h>
#include <sys/signalfd.h>

#include <audiomanagerconfig.h>
#include "audiomanagertypes.h"

#ifdef WITH_TIMERFD
# include <stdio.h>
# include <string>
# include <stdexcept>
# include <fcntl.h>
#endif // ifdef WITH_TIMERFD

namespace am
{

#define MAX_NS          1000000000L
#define MAX_TIMERHANDLE UINT16_MAX
#define MAX_POLLHANDLE  UINT16_MAX

typedef uint16_t        sh_pollHandle_t;  //!< this is a handle for a filedescriptor to be used with the SocketHandler
typedef sh_pollHandle_t sh_timerHandle_t; //!< this is a handle for a timer to be used with the SocketHandler

/**
 * prototype for poll prepared callback
 */
class IAmShPollPrepare
{
public:
    virtual void Call(const sh_pollHandle_t handle, void *userData) = 0;

    virtual ~IAmShPollPrepare() {}
};

/**
 * prototype for poll fired callback
 */
class IAmShPollFired
{
public:
    virtual void Call(const pollfd pollfd, const sh_pollHandle_t handle, void *userData) = 0;

    virtual ~IAmShPollFired() {}
};

/**
 * prototype for poll check callback
 */
class IAmShPollCheck
{
public:
    virtual bool Call(const sh_pollHandle_t handle, void *userData) = 0;

    virtual ~IAmShPollCheck() {}
};

/**
 * prototype for dispatch callback
 */
class IAmShPollDispatch
{
public:
    virtual bool Call(const sh_pollHandle_t handle, void *userData) = 0;

    virtual ~IAmShPollDispatch() {}
};

/**
 * prototype for the timer callback
 */
class IAmShTimerCallBack
{
public:
    IAmShTimerCallBack(){}
    virtual void Call(const sh_timerHandle_t handle, void *userData) = 0;

    virtual ~IAmShTimerCallBack(){}
};

/**make private, not public
 * template for a callback
 */
template<class TClass>
class TAmShPollFired : public IAmShPollFired
{
private:
    TClass *mInstance;
    void    (TClass::*mFunction)(const pollfd pollfd, const sh_pollHandle_t handle, void *userData);

public:
    TAmShPollFired(TClass *instance, void (TClass::*function)(const pollfd pollfd, const sh_pollHandle_t handle, void *userData))
        : mInstance(instance)
        , mFunction(function)
    {}

    virtual void Call(const pollfd pollfd, const sh_pollHandle_t handle, void *userData)
    {
        (*mInstance.*mFunction)(pollfd, handle, userData);
    }

};

/**
 * template for a callback
 */
template<class TClass>
class TAmShPollCheck : public IAmShPollCheck
{
private:
    TClass *mInstance;
    bool    (TClass::*mFunction)(const sh_pollHandle_t handle, void *userData);

public:
    TAmShPollCheck(TClass *instance, bool (TClass::*function)(const sh_pollHandle_t handle, void *userData))
        : mInstance(instance)
        , mFunction(function)
    {}

    virtual bool Call(const sh_pollHandle_t handle, void *userData)
    {
        return ((*mInstance.*mFunction)(handle, userData));
    }

};

/**
 * template for a callback
 */
template<class TClass>
class TAmShPollDispatch : public IAmShPollDispatch
{
private:
    TClass *mInstance;
    bool    (TClass::*mFunction)(const sh_pollHandle_t handle, void *userData);

public:
    TAmShPollDispatch(TClass *instance, bool (TClass::*function)(const sh_pollHandle_t handle, void *userData))
        : mInstance(instance)
        , mFunction(function)
    {}

    virtual bool Call(const sh_pollHandle_t handle, void *userData)
    {
        return ((*mInstance.*mFunction)(handle, userData));
    }

};

/**
 * template to create the functor for a class
 */
template<class TClass>
class TAmShTimerCallBack : public IAmShTimerCallBack
{
private:
    TClass *mInstance;
    void    (TClass::*mFunction)(sh_timerHandle_t handle, void *userData);

public:
    TAmShTimerCallBack(TClass *instance, void (TClass::*function)(sh_timerHandle_t handle, void *userData))
        : IAmShTimerCallBack()
        , mInstance(instance)
        , mFunction(function)
    {}

    virtual void Call(sh_timerHandle_t handle, void *userData)
    {
        (*mInstance.*mFunction)(handle, userData);
    }

};

/**
 * template for a callback
 */

template<class TClass>
class TAmShPollPrepare : public IAmShPollPrepare
{
private:
    TClass *mInstance;
    void    (TClass::*mFunction)(const sh_pollHandle_t handle, void *userData);

public:
    TAmShPollPrepare(TClass *instance, void (TClass::*function)(const sh_pollHandle_t handle, void *userData))
        : mInstance(instance)
        , mFunction(function)
    {}

    virtual void Call(const sh_pollHandle_t handle, void *userData)
    {
        (*mInstance.*mFunction)(handle, userData);
    }

};

/**
 *  @class CAmSocketHandler
 *
 *  The am::CAmSocketHandler is the implementation of the main event dispatching loop
 *  used in AudioManager daemon (service) and available for use in its plug-ins as well
 *  as remote client applications communicating with the daemon through the command
 *  or routing interface.
 *  The general concept is described on page @ref mainl
 *
 *  The AM daemon itself launches a single socket handler instance. Plugins can access
 *  it through the getSocketHandler() function provided by each of the public interfaces.
 *  On the other hand, remote applications have to setup their own instance.
 *
 *  Modules interested in being informed on incidents open their input (referenced as file
 *  descriptor fd) and register a related callback function to the CAmSocketHandler instance.
 *  There also exist means to register callback functions for elapsed timers and captured signals.
 *
 *  The loop does not start automatically on object creation. Instead it is started via
 *  am::CAmSocketHandler::mainLoop() and its self-termination is triggered via
 *  am::CAmSocketHandler::exit_mainloop().
 *  Example code can be found in am::CAmDbusWrapper.
 *
 *  ### Descriptor states #
 *
 *  Internally each registered file descriptor (fd) is marked with a state indicator. This
 *  can be any of these values:
 *     | Value   | Description                                         |
 *     | :-----: | :-------------------------------------------------- |
 *     | ADD     | newly announced fd scheduled for monitoring         |
 *     | UPDATE  | ready for monitoring (fd is either new or altered)  |
 *     | VALID   | fd is actively monitored                            |
 *     | REMOVE  | fd marked for removal                               |
 *
 *  Only descriptors marked as VALID are monitored by ppoll(). State transitions are
 *  defined as below:
 *
 *  @dot  "States of registered File Descriptors"
 *    digraph {
 *      announce           [rank=source]
 *      VALID              [style=filled]
 *      dropped            [rank=sink]
 *
 *      // normal flow - registration
 *      announce-> ADD     [label="addFDPoll()"]
 *      ADD     -> UPDATE  [label="mainLoop()"]
 *      UPDATE  -> VALID   [label="mainLoop()"]
 *
 *      // normal flow - deregistration
 *      VALID   -> REMOVE  [label="removeFDPoll()"]
 *      REMOVE  -> dropped [label="mainLoop()"]
 *
 *      // other transitions
 *      edge [color=grey]
 *      ADD     -> dropped [label="removeFDPoll()"]
 *      UPDATE  -> REMOVE  [label="removeFDPoll()"]
 *      VALID   -> UPDATE  [label="updateEventFlags()"]
 *      REMOVE  -> UPDATE  [label="addFDPoll()"]
 *  }
 *  @enddot
 *
 *  ### Multi-Thread Support #
 *
 *  Besides the above mentioned single-threaded usage where all inputs of interest
 *  are registered before the mainLoop() is started and deregistered after its termination,
 *  @ref am::CAmSocketHandler "CAmSocketHandler" also supports dynamic invocation of
 *
 *    - addFDPoll() / removeFDPoll()
 *    - addTimer() / removeTimer()
 *    - addSignalHandler() removeSignalHandler()
 *
 *  Therefore a common, mutex-based helper class @ref am::CAmSocketHandler::AutoLock "AutoLock"
 *  protects all accesses to the internal data structures. This is sufficient for all registrations.
 *
 *  On dynamic deregistration, it is essential that the externally hosted callback functions
 *  are not invoked again after their deregistration (and possible destruction). Therefore,
 *  additionally the return of the corresponding remove...() function is delayed until no
 *  further invocation of the registered callback function is neither ongoing nor scheduled.
 *  This is ensured by individual instances of the semaphore-based sub-class @ref am::CAmSocketHandler::AutoSync "AutoSync".
 *
 *  Internally, the deregistration handling slightly differs between socket-like inputs
 *  and signals/timers.
 *    1. When removing socket-like file descriptors, the removeFDPoll() function
 *       creates a temporary AutoSync instance, initialized with 0, and waits until
 *       the mainLoop signals that it has finished working on it.
 *    2. removeSignalHandler() waits for the AutoSync instance permanently associated
 *       with the callback function. Since the AutoSync object in this case is initialized
 *       with 1, the wait() will return immediately unless the mainLoop is in progress
 *       of invoking the callback.
 *    3. For the handling of removeTimer the implementation depends on build option
 *       WITH_TIMERFD. For WITH_TIMERFD=ON, the protection is covered by the mechanism
 *       implemented for socket-like file descriptors (1.). In case WITH_TIMERFD=OFF,
 *       the mechanism is implemented identically to above removeSignalHandler().
 */
class CAmSocketHandler
{
    typedef enum : uint8_t
    {
        ADD     = 0u, // new, uninitialized element which needs to be added to ppoll array
        UPDATE  = 1u, // update of event information therefore update ppoll array
        VALID   = 2u, // it is a valid element in ppoll array
        REMOVE  = 3u  // remove from ppoll array and internal map
    } poll_states_e;

    // private helper class to block return from removeFDPoll()
    // and exit_mainloop() until mainLoop-thread is garanteed to
    // not call involved callbacks any more
    class AutoSync;

    struct sh_poll_s //!< struct that holds information about polls
    {
        sh_pollHandle_t handle   = 0;                                                                         //!< handle to uniquely adress a filedesriptor
        pollfd          pollfdValue;                                                                             //!< the array for polling the filedescriptors
        std::function<void(const sh_pollHandle_t handle, void *userData)> prepareCB;                    // preperation callback
        std::function<void(const pollfd pollfd, const sh_pollHandle_t handle, void *userData)> firedCB; // fired callback
        std::function<bool(const sh_pollHandle_t handle, void *userData)> checkCB;                      // check callback
        std::function<bool(const sh_pollHandle_t handle, void *userData)> dispatchCB;                   // dispatch callback
        void           *userData = nullptr;
        poll_states_e   state    = ADD;
        AutoSync       *pSync    = nullptr;
    };

    struct sh_timer_s //!< struct that holds information of timers
    {
        sh_timerHandle_t handle    = 0;       //!< the handle of the timer
#ifdef WITH_TIMERFD
        int              fd        = -1;
        itimerspec       countdown;           //!< the countdown, this value is decreased every time the timer is up
#else
        timespec         countdown = {0, 0};  //!< the countdown, this value is decreased every time the timer is up
        AutoSync        *pSync     = nullptr; //!< coordinate usage with removal
#endif
        std::function<void(const sh_timerHandle_t handle, void *userData)>
                         callback;            //!< timer callback
        void            *userData  = nullptr; //!< pointer to data needed in callback
    };

    struct sh_signal_s
    {
        sh_pollHandle_t handle   = 0;       //!< identifier to uniquely address a signal handler
        std::function<void(const sh_pollHandle_t handle, const signalfd_siginfo &info, void *userData)>
                        callback;           //!< user-provided callback function
        void           *userData = nullptr; //!< arbitrary user object providing all information needed in callback function
        AutoSync       *pSync    = nullptr; //!< coordinate usage with removal
    };

    struct sh_identifier_s
    {
        std::set<sh_pollHandle_t> pollHandles;
        uint16_t limit;
        uint16_t lastUsedID;
        sh_identifier_s(const uint16_t pollLimit = UINT16_MAX)
            : pollHandles()
            , limit(pollLimit)
            , lastUsedID(0)
        {}
    };

    typedef std::map<int, sh_poll_s>          MapShPoll_t;            //!< list for the callbacks
    typedef std::vector<sh_signal_s>          VectorSignalHandlers_t; //!< list for the callbacks

    typedef enum : uint8_t
    {
        NO_ERROR = 0u,    // OK
        FD_ERROR = 1u,    // Invalid file descriptor
        MT_ERROR = 2u     // Multi-thread issue
    } internal_codes_e;
    typedef uint8_t internal_codes_t;

    // protector for multi-threaded access to below internal containers:
    //  - mListActiveTimer
    //  - mListTimer
    //  - mMapShPoll
    //  - mSetPollKeys
    //  - mSetSignalhandlerKeys
    //  - mSetTimerKeys
    //  - mSignalHandlers
    pthread_mutex_t        mMutex;
    class AutoLock;

    pid_t                  mLoopThread;   // thread-id of mainLoop to check whether thread-sync is required
    std::list<AutoSync *>  mExitSyncs;    // synchronize exit_mainloop() until mainLoop has terminated,
                                          //   even if multiple invocations are pending.
    int                    mEventFd;
    int                    mSignalFd;
    enum
    {
        DOWN,   // not started or shut down completed
        UP,     // running
        QUIT    // scheduled for shut down
    }                      mLoopStatus;           // status of the main loop

    MapShPoll_t            mMapShPoll;            // list that holds all information for the ppoll

    sh_identifier_s        mSetPollKeys;          // A set of all used ppoll keys
    sh_identifier_s        mSetTimerKeys;         // A set of all used timer keys
    std::list<sh_timer_s>  mListTimer;            // list of all timers
    sh_identifier_s        mSetSignalhandlerKeys; // A set of all used signal handler keys
    VectorSignalHandlers_t mSignalHandlers;
    internal_codes_t       mInternalCodes;
#ifndef WITH_TIMERFD
    std::list<sh_timer_s>  mListActiveTimer;      // list of all currently active timers
    timespec               mStartTime;            // here the actual time is saved for timecorrection
#endif

private:
    bool fdIsValid(const int fd) const;
    void wakeupWorker(const std::string &func);

    timespec *insertTime(timespec &buffertime);

#ifdef WITH_TIMERFD
    am_Error_e createTimeFD(const itimerspec &timeouts, int &fd);

#else
    void timerUp();
    void timerCorrection();

    /**
     * compares countdown values
     * @param a
     * @param b
     * @return true if b greater a
     */
    inline static bool compareCountdown(const sh_timer_s &a, const sh_timer_s &b)
    {
        return ((a.countdown.tv_sec == b.countdown.tv_sec) ? (a.countdown.tv_nsec < b.countdown.tv_nsec) : (a.countdown.tv_sec < b.countdown.tv_sec));
    }

    /**
     * Subtracts b from a
     * @param a
     * @param b
     * @return subtracted value
     */
    inline static timespec timespecSub(const timespec &a, const timespec &b)
    {
        timespec result;

        if ((a.tv_sec < b.tv_sec) || ((a.tv_sec == b.tv_sec) && (a.tv_nsec <= b.tv_nsec)))
        {
            result.tv_sec = result.tv_nsec = 0;
        }
        else
        {
            result.tv_sec = a.tv_sec - b.tv_sec;
            if (a.tv_nsec < b.tv_nsec)
            {
                result.tv_nsec = a.tv_nsec + MAX_NS - b.tv_nsec;
                result.tv_sec--; /* Borrow a second. */
            }
            else
            {
                result.tv_nsec = a.tv_nsec - b.tv_nsec;
            }
        }

        return (result);
    }

    /**
     * adds timespec values
     * @param a
     * @param b
     * @return the added values
     */
    inline timespec timespecAdd(const timespec &a, const timespec &b)
    {
        timespec result;
        result.tv_sec  = a.tv_sec + b.tv_sec;
        result.tv_nsec = a.tv_nsec + b.tv_nsec;
        if (result.tv_nsec >= MAX_NS)
        {
            result.tv_sec++;
            result.tv_nsec = result.tv_nsec - MAX_NS;
        }

        return (result);
    }

    /**
     * comapares timespec values
     * @param a
     * @param b
     * @return
     */
    inline int timespecCompare(const timespec &a, const timespec &b)
    {
        // less
        if (a.tv_sec < b.tv_sec)
        {
            return (-1);
        }
        // greater
        else if (a.tv_sec > b.tv_sec)
        {
            return (1);
        }
        // less
        else if (a.tv_nsec < b.tv_nsec)
        {
            return (-1);
        }
        // greater
        else if (a.tv_nsec > b.tv_nsec)
        {
            return (1);
        }

        // equal
        return (0);
    }
#endif  // ifdef WITH_TIMERFD

    /**
     * functor to prepare all fire events
     * @param a
     * @return
     */
    inline static void prepare(sh_poll_s &row);

    /**
     * functor to return all fired events
     * @param a
     * @return
     */
    inline static void fire(const sh_poll_s &a);

    /**
     * timer fire callback
     * @param a
     * @return
     */
    inline static void callTimer(sh_timer_s &a);

    /**
     * next handle id
     * @param std::set handles
     * @return handle
     */
    bool nextHandle(sh_identifier_s &handle);

public:

    CAmSocketHandler();
    ~CAmSocketHandler();

    /**
     * install the signal fd
     */
    am_Error_e listenToSignals(const std::vector<uint8_t> &listSignals);

    am_Error_e addFDPoll(const int fd, const short event, std::function<void(const sh_pollHandle_t handle, void *userData)> prepare, std::function<void(const pollfd pollfd, const sh_pollHandle_t handle, void *userData)> fired,
        std::function<bool(const sh_pollHandle_t handle, void *userData)> check, std::function<bool(const sh_pollHandle_t handle, void *userData)> dispatch, void *userData, sh_pollHandle_t &handle);

    am_Error_e addFDPoll(const int fd, const short event, IAmShPollPrepare *prepare, IAmShPollFired *fired, IAmShPollCheck *check, IAmShPollDispatch *dispatch, void *userData, sh_pollHandle_t &handle);
    am_Error_e removeFDPoll(const sh_pollHandle_t handle);
    am_Error_e updateEventFlags(const sh_pollHandle_t handle, const short events);

    am_Error_e addSignalHandler(std::function<void(const sh_pollHandle_t handle, const signalfd_siginfo &info, void *userData)> callback, sh_pollHandle_t &handle, void *userData);
    am_Error_e removeSignalHandler(const sh_pollHandle_t handle);

    am_Error_e addTimer(const timespec &timeouts, IAmShTimerCallBack *callback, sh_timerHandle_t & handle, void *userData,
#ifndef WITH_TIMERFD
        const bool __attribute__((__unused__)) repeats = false
#else
        const bool repeats = false
#endif
        );
    am_Error_e addTimer(const timespec &timeouts, std::function<void(const sh_timerHandle_t handle, void *userData)> callback, sh_timerHandle_t & handle, void *userData,
#ifndef WITH_TIMERFD
        const bool __attribute__((__unused__)) repeats = false
#else
        const bool repeats = false
#endif
        );
    am_Error_e removeTimer(const sh_timerHandle_t handle);
    am_Error_e restartTimer(const sh_timerHandle_t handle);
    am_Error_e updateTimer(const sh_timerHandle_t handle, const timespec &timeouts);
    am_Error_e stopTimer(const sh_timerHandle_t handle);

    /**
     *  Continuous, blocking listener to incidents on registered file descriptors.
     *  Self-terminates once mLoopStatus is set to QUIT
     */
    void mainLoop(void);
    /** legacy alias for mainLoop() */ inline void start_listenting()  {  mainLoop();  }

    /**
     *  Trigger self-termination of mainLoop() by setting mLoopStatus to QUIT
     */
    void exit_mainloop();
    /** legacy alias for exit_mainloop(); **/  inline void stop_listening()  {  exit_mainloop();  }

    bool fatalErrorOccurred();

};

} /* namespace am */
#endif /* SOCKETHANDLER_H_ */
